// validations/kycSessionValidation.js
const Joi = require('joi');

// Schema for creating a new KYC session with custom messages
const kycSessionValidationSchema = Joi.object({
  tenantId: Joi.string()
    .required()
    .pattern(/^[0-9a-fA-F]{24}$/)
    .messages({
      'any.required': 'Tenant ID is required',
      'string.empty': 'Tenant ID cannot be empty',
      'string.pattern.base': 'Tenant ID must be a valid MongoDB ObjectId'
    }),
    
  customerId: Joi.string()
    .required()
    .pattern(/^[0-9a-fA-F]{24}$/)
    .messages({
      'any.required': 'Customer ID is required',
      'string.empty': 'Customer ID cannot be empty',
      'string.pattern.base': 'Customer ID must be a valid MongoDB ObjectId'
    }),
    
  fiUserId: Joi.string()
    .required()
    .pattern(/^[0-9a-fA-F]{24}$/)
    .messages({
      'any.required': 'FI User ID is required',
      'string.empty': 'FI User ID cannot be empty',
      'string.pattern.base': 'FI User ID must be a valid MongoDB ObjectId'
    }),
    
  officerId: Joi.string()
    .optional()
    .pattern(/^[0-9a-fA-F]{24}$/)
    .allow(null, '')
    .messages({
      'string.pattern.base': 'Officer ID must be a valid MongoDB ObjectId'
    }),
    
  sessionStart: Joi.date()
    .optional()
    .default(() => new Date())
    .messages({
      'date.base': 'Session start time must be a valid date'
    }),
    
  sessionEnd: Joi.date()
    .optional()
    .greater(Joi.ref('sessionStart'))
    .allow(null)
    .messages({
      'date.base': 'Session end time must be a valid date',
      'date.greater': 'Session end time must be after session start time'
    }),
    
  status: Joi.string()
    .optional()
    .valid('initiated', 'in_review', 'approved', 'rejected')
    .default('initiated')
    .messages({
      'any.only': 'Status must be one of: initiated, in_review, approved, or rejected'
    }),
    
  rejectionReason: Joi.string()
    .optional()
    .when('status', {
      is: 'rejected',
      then: Joi.required().messages({
        'any.required': 'Rejection reason is required when status is rejected'
      }),
      otherwise: Joi.optional()
    })
    .max(500)
    .allow(null, '')
    .messages({
      'string.max': 'Rejection reason cannot exceed 500 characters'
    })
});

// Schema for updating KYC session
const updateKycSessionValidationSchema = Joi.object({
  tenantId: Joi.string()
    .optional()
    .pattern(/^[0-9a-fA-F]{24}$/)
    .messages({
      'string.pattern.base': 'Tenant ID must be a valid MongoDB ObjectId'
    }),
    
  customerId: Joi.string()
    .optional()
    .pattern(/^[0-9a-fA-F]{24}$/)
    .messages({
      'string.pattern.base': 'Customer ID must be a valid MongoDB ObjectId'
    }),
    
  fiUserId: Joi.string()
    .optional()
    .pattern(/^[0-9a-fA-F]{24}$/)
    .messages({
      'string.pattern.base': 'FI User ID must be a valid MongoDB ObjectId'
    }),
    
  officerId: Joi.string()
    .optional()
    .pattern(/^[0-9a-fA-F]{24}$/)
    .allow(null, '')
    .messages({
      'string.pattern.base': 'Officer ID must be a valid MongoDB ObjectId'
    }),
    
  sessionStart: Joi.date()
    .optional()
    .messages({
      'date.base': 'Session start time must be a valid date'
    }),
    
  sessionEnd: Joi.date()
    .optional()
    .allow(null)
    .messages({
      'date.base': 'Session end time must be a valid date'
    }),
    
  status: Joi.string()
    .optional()
    .valid('initiated', 'in_review', 'approved', 'rejected')
    .messages({
      'any.only': 'Status must be one of: initiated, in_review, approved, or rejected'
    }),
    
  rejectionReason: Joi.string()
    .optional()
    .max(500)
    .allow(null, '')
    .messages({
      'string.max': 'Rejection reason cannot exceed 500 characters'
    })
});

// Middleware to validate create KYC session payload
function validateKycSession(req, res, next) {
  const { error, value } = kycSessionValidationSchema.validate(req.body, { 
    abortEarly: false,
    stripUnknown: true,
    convert: true
  });
  
  if (error) {
    return res.status(400).json({
      success: false,
      error: 'KYC session validation failed',
      code: 'VALIDATION_ERROR',
      details: error.details.map(detail => ({
        field: detail.path.join('.'),
        message: detail.message,
        value: detail.context?.value
      }))
    });
  }
  
  req.body = value;
  next();
}

// Middleware to validate update KYC session payload
function updateValidateKycSession(req, res, next) {
  const { error, value } = updateKycSessionValidationSchema.validate(req.body, { 
    abortEarly: false,
    stripUnknown: true,
    convert: true
  });
  
  if (error) {
    return res.status(400).json({
      success: false,
      error: 'KYC session update validation failed',
      code: 'UPDATE_VALIDATION_ERROR',
      details: error.details.map(detail => ({
        field: detail.path.join('.'),
        message: detail.message,
        value: detail.context?.value
      }))
    });
  }
  
  req.body = value;
  next();
}

// Additional validation for status transitions
// function validateStatusTransition(req, res, next) {
//   const { status } = req.body;
  
//   if (status) {
//     const allowedTransitions = {
//       'initiated': ['in_review', 'rejected'],
//       'in_review': ['approved', 'rejected'],
//       'approved': [], // No transitions allowed from approved
//       'rejected': ['initiated'] // Can restart process
//     };
    
//     // This would require fetching current status, but for now just validate format
//     const validStatuses = ['initiated', 'in_review', 'approved', 'rejected'];
//     if (!validStatuses.includes(status)) {
//       return res.status(400).json({
//         success: false,
//         error: 'Invalid status transition',
//         code: 'INVALID_STATUS_TRANSITION',
//         allowedStatuses: validStatuses
//       });
//     }
//   }
  
//   next();
// }

module.exports = {
    validateKycSession,
    updateValidateKycSession
}
